/* 	MGV136 Versie Gerard    8 March 2016


The code below will control the servos of the MGV136 and the relay of the MGV137.
Input signals come from a MGV50 by means of four signal wires.
When the variable "Program_mode" == 0 then the 4 signal wires just give you the state of the 4 servos.
The relay switch when the servo has reached its end position.
The relays are on a so-called "push" PCB and thus communication is via the serial connector 480 baud. 480 is not a typo!

In order to be able to change the positions of servos and relays in Rocrail there is the possibility to go to the Program Mode.
Rocrail sends through the four signal wires a cycle of numbers that will be recognized by the program below.
Once completed this cycle the variable "program_mode" == 1 and you can adjust the positions of servos and relay via Rocrail.


 ******************************************************************************
 * Standard include files
 ******************************************************************************
 */


 

// #pragma config statements should precede project file includes.
// Use project enums instead of #define for ON and OFF.
#include <xc.h> 
#include <pic.h>
#include <stdio.h>
#include <pic16F648A.h>
#include "eeprom.h"

#pragma config FOSC = INTOSCIO  // Oscillator Selection bits (INTOSC oscillator: I/O function on RA6/OSC2/CLKOUT pin, I/O function on RA7/OSC1/CLKIN)
#pragma config WDTE = OFF       // Watchdog Timer Enable bit (WDT disabled)
//#pragma config PWRTE = ON       // Power-up Timer Enable bit (PWRT enabled), werkt niet tijdens debug
#pragma config MCLRE = ON       // RA5/MCLR/VPP Pin Function Select bit (RA5/MCLR/VPP pin function is MCLR)
#pragma config BOREN = ON       // Brown-out Detect Enable bit (BOD enabled)
#pragma config LVP = OFF        // Low-Voltage Programming Enable bit (RB4/PGM pin has digital I/O function, HV on MCLR must be used for programming)
#pragma config CPD = OFF        // Data EE Memory Code Protection bit (Data memory code protection off)
#pragma config CP = OFF         // Flash Program Memory Code Protection bit (Code protection off)
#define _XTAL_FREQ 4000000 

/*
 *******************************************************************************************
 * Macro definitions
 *******************************************************************************************
 */
//#define Checkbit    (adres, bit) 	(adres &   (1 <<  bit))			// Controleert of een bit == 1.    Example: if (Checkbit (PIND, 5) { .......} 
#define setbit      (port, bit) 	(port) |=  (1 << (bit))
#define clearbit    (port, bit) 	(port) &= ~(1 << (bit))
#define toggle_bit  (port, bit)		(port) ^=  (1 << (bit))		

#define Puls_1msec								1000		// For 1    msec the preset of Timer1 should be 1000 counts 	65536 - 1000 = 64536
#define Puls_1_5_msec							1500		// Voor 1,5 msec the preset of Timer1 should be 1500 counts     65536 - 1500 = 64036
#define Puls_2msec								2000		// Voor 2   msec the preset of Timer1 should be 2000 counts     65536 - 2000 = 63536
#define stap_timer								11			// Determines how quickly the servo moves


// *******************************************************************************************
// * Variables
// *******************************************************************************************

// EEprom start addresses
const unsigned char	  eePromLeeg_b      = 0;   // 1 byte.  When eePromLeeg_b != 1 then the default values are read into EEPROM.

const unsigned char	  eeServoRichting_b = 1;   // 4 bytes. This is the final direction of the servo, just before the power was turned off.
const unsigned char	  eeServoStap_b     = 5;   // 4 bytes. Increment per servo.	
const unsigned char   eeServoRelais_b   = 9;   // 4 bytes. When '1' then the user has reversed the relay
const unsigned int    eeServoStand_w    = 13;  // 8 bytes. eeServoStand_w contains the latest state of the servo, just before the power was turned off.
const unsigned int    eeServoLinks_w    = 21;  // 8 bytes. Leftmost  position of the servo
const unsigned int    eeServoRechts_w   = 29;  // 8 bytes. Rightmost position of the servo
const unsigned int    eeServoBounce_b   = 37;  // 4 bytes. Bounce active ?  Bounce > 0 ?

typedef struct
{
	unsigned char   richting;				// current direction of the servo during program flow
	unsigned int    linksom;				// Leftmost   position of the servo
	unsigned int    rechtsom;				// Rightmost  position of the servo
	unsigned int    stand;					// latest state of the servo
	unsigned char   stapgrootte;			// Increment per servo
	unsigned char   eindstand;				// When eindstand == 1 this means that the servo is at one of the two limits. The program will bring the control signal to 5V
	unsigned char   relais;                 // Fill in "0" or "1" will reverse polarity relay
    unsigned char   bounce;                 // Bounce active ? J/N  1/0
    unsigned char   bounceCycle;            // keeps track of the bounce-cycle
	
}Servo_info;

Servo_info servo[4];             // variabele voor servo's, 0 t/m 3,  

#define INPUT1    RA4  
#define INPUT2    RA6   
#define INPUT3    RA7   
#define INPUT4    RB0   
#define LE1       RB3  
#define FEEDBACK1 RA0 
#define FEEDBACK2 RA1  
#define FEEDBACK3 RA2  
#define FEEDBACK4 RA3
#define Servo1    RB4
#define Servo2    RB5    
#define Servo3    RB6    
#define Servo4    RB7    

unsigned char programmingSequence[] = { 1, 3, 7, 15, 14, 12, 8, 0 };
// After receiving this sequence on the input the program goes into Servo Program mode for adjusting values of the servo

unsigned char enterProgramState=0;	// This counter keeps track of how far the array programming Sequence [] has already been completed. 
                                    // When enterProgramState == 7, the program goes into program mode.
unsigned char programMode=0;    	// When programMode == 1 than the software is in Program Mode.
unsigned char input;				// Variabele "input" shows the value on the input 
unsigned char target=0;				// Temporarily servo number
unsigned char program_blck = 0;


//*************** Declaratie van functies ********************
void Servo_uitgang_hoog (unsigned char nr);
void Servo_tijden (unsigned char sv_nr, unsigned char li_re);
void ReadNibble();
void WaitNS(unsigned int nano_sec);
void Schakel_relais_uitgang ();
void Feedback (unsigned char sv_nr, unsigned char status);
//************************************************************

/**
******************************************************
Functie         void Init (void)
Beschrijving    Deze functie begint met alle hardware instellingen van de controller op de juiste manier in te stellen
            	Wanneer de EEPROM geen data blijkt te hebben dan wordt de EEPROM gevuld met standaard waarden
				Als laatste worden de verschillende variabelen gevuld met de data die in de EEPROM staat
ingang_var		geen
retour_var		geen
*****************************************************
**/

void Init()
{
	unsigned char i;
	
    CMCON  = 0x07;              // Turn comparators off and enable pins for I/O functions
 	TRISA  = 0xF0;              // initialize Port A
    TRISB  = 0x07;              // initialize Port B
    BRGH = 0; 
	SPBRG = 129;                // 480 Baud 
	/* Enable serial port */ 
	SYNC = 0;                   // Asynchronous mode
	SPEN = 1;                   // Serial Port enabled
	TXEN = 1;                   // Transmit enabled

    // Timer1 Registers: Prescaler=1:1; Free running timer at Fosc / 4 == 1 MHz

    T1CKPS1 = 0;  // bits 5-4  Prescaler Rate 1:1
    T1CKPS0 = 0;  // 
    T1OSCEN = 0;  // bit 3 Timer1 Oscillator Enable Control: MUST BE OFF, otherwise the Timer / program will act very slow
    TMR1CS  = 0;  // bit 1 Timer1 Clock Source Select bit:0=Internal clock (FOSC/4)
    TMR1ON  = 0;  // bit 0 disable timer
    TMR1H = 0;    // preset for timer1 MSB register
    TMR1L = 0;    // preset for timer1 LSB register
    
    Servo1 = 1;     // All servo outputs high
    Servo2 = 1;
    Servo3 = 1;
    Servo4 = 1;
    LE1    = 0;     // program LED off

//	The library functions eeprom_read() and eeprom_write(), can be called to read from, and write to, the EEPROM during program execution.
//  Ondanks dat onderstaande code werkt is het misschien wel een goed idee om dit nog eens te bekijken.    
//  Zie http://microchip.wikidot.com/faq:36
    
    
    if (readEEprom(eePromLeeg_b) != 1)    // if the first byte != 1 than the EEPROM is filled with default values
	{
		for (i=0; i<4; ++i)          
		{
            writeEEprom((eeServoRichting_b + i) ,1);
  			writeEEprom((eeServoStap_b + i), stap_timer );			// startwaarde voor de stapgrootte is 8
            writeEEprom((eeServoRelais_b + i) ,0);
            eeprom_write_word((eeServoStand_w + (2*i)), Puls_1_5_msec);		// middenstand als startwaarde.   + (2*i) komt omdat het hier om een word variabele gaat.
			eeprom_write_word((eeServoLinks_w + (2*i)), Puls_2msec);		// eindstand links
			eeprom_write_word((eeServoRechts_w +(2*i)), Puls_1msec);		// eindstand rechts
            writeEEprom((eeServoBounce_b + i), 0);                          // Bounce not active
		}  
		writeEEprom(eePromLeeg_b, 1);  // De eeprom is nu gevuld met gegevens voor de volgende spannings onderbreking
	}
	
	
	for (i=0; i<4; ++i)		// fill the variables with the values from EEPROM
	{
        
		servo[i].richting		= readEEprom((eeServoRichting_b + i));
		servo[i].stapgrootte	= readEEprom((eeServoStap_b + i));
        servo[i].relais			= readEEprom((eeServoRelais_b + i));	// Wanneer '1' dan heeft de gebruiker het relais omgepoold
        servo[i].stand			= eeprom_read_word((eeServoStand_w  + (2*i)));
		servo[i].linksom		= eeprom_read_word((eeServoLinks_w  + (2*i)));
		servo[i].rechtsom		= eeprom_read_word((eeServoRechts_w + (2*i)));
    	servo[i].bounce         = readEEprom((eeServoBounce_b + i));
        servo[i].eindstand      = 1;
        servo[i].bounce         = 3;
        servo[i].bounceCycle    = servo[i].bounce;
        
        
        Feedback (i, servo[i].richting);        // All output feedback signals to the right value
    }   
    Schakel_relais_uitgang();                   // Initialize the relais after power up
      
}	

/**
******************************************************
Functie         static void ReadInput()
Beschrijving    Leest 4 bits van de 4 datalijnen.  De data wordt in de variabele "input" gezet.
				Controleert of dat Rocrail naar program-mode wil of dat er gewoon een servo + relais omgezet moet worden
				Om in de programmode te komen stuurt Rocrail achter elkaar een sequence van getallen via de ingangen,  de tijd tussen de verschillende getallen is ongeveer 7 - 8 msec
				Wanneer het getal aan de ingang wel past binnen de sequence maar een gewone wisselstand is dan wordt de "counter" naar 0 geteld.  Dit duurt ongeveer 14 msec.
				Om in programmode te komen is Rocrail dus sneller dan het aftellen van de counter.
ingang_var		geen
retour_var		globale variabele "input"
*****************************************************
**/

static void ReadInput()
{
	static unsigned int counter;
	static unsigned int i;
	
	
	while (1)
	{													// lees de 4 ingangen en zet de waarde in variabele input
		input = 0;										// variabele input begint bij 0
		if (INPUT1) { input |= 0x01; }
		if (INPUT2) { input |= 0x02; }
		if (INPUT3) { input |= 0x04; }
		if (INPUT4) { input |= 0x08; }
			
		if (input == 0) { program_blck = 0; }			// Wanneer 'input' is 0000 dan wil Rocrail misschien weer naar programmode ==> program_blck = weer 0
		
		if ((input == programmingSequence[enterProgramState]) && (program_blck == 0))
		{
			if (enterProgramState == 7)					// Sequence van getallen klopt, Rocrail wil naar Program-mode
			{
				programMode = 1;
				enterProgramState = 0;
				LE1 = 1;                                // switch on led LE1 so the user knows the program-mode is active
				return;                                 // leave this function
			}
			else
			{
				enterProgramState++;
				counter = 200;                          // start waiting loop
			}
		}
		else if ((enterProgramState != 0) && (input == programmingSequence[enterProgramState-1]))
		{
			// Found input for the previous state, do nothing
			// if the counter reaches 0, we'll go out of the routine.
		}
		else
		{
			// Input is not a part of the programming sequence.
			goto no_programming_sequence;
		}
		
		// Bail out if counter reached 0.
		if (counter == 0)
		{
			// Rocrail stuurt zijn informatie heel snel, wanneer 'counter == 0' dan was het een gewoon verzoek om een servo om te zetten.
			program_blck = 1;					// Zolang 'program_blck == 1' wordt er niet meer gekeken of Rocrail naar programmode wil.
			goto no_programming_sequence;
		}
		else
		{
			counter--;
			
			i = 200;		// just wait long enough
			while (i--);
		}
	}
	
	no_programming_sequence:
	enterProgramState = 0;
	
	return;
}

/**
******************************************************
Functie         void ReadNibble (void)
Beschrijving    Leest een 4 - bits nibble. De MSB bit is de schrijflijn waarmee aangegeven wordt dat de overige 3 bits de juiste data bevatten
            	De data wordt in de variabele "input" gezet.
				Deze functie is alleen actief wanner het programma in de Program mode staat
ingang_var		geen
retour_var		globale variabele "input"
*****************************************************
**/

void ReadNibble()
{
	while (INPUT4 == 0 ) {}			// wachten totdat de schrijflijn actief is
	
	// Nu kunnen we de ingangen uitlezen
	input = 0;
	if (INPUT1) { input |= 0x01; }
	if (INPUT2) { input |= 0x02; }
	if (INPUT3) { input |= 0x04; }
	
	while (INPUT4 != 0 ) {}  			// wachten totdat de schrijflijn weer inactief is
}

/**
******************************************************
Functie			void ReadPositionNibble()
Beschrijving    Leest twee 4 - bits nibble. De MSB bit is de schrijflijn waarmee aangegeven wordt dat de overige 3 bits de juiste data bevatten
            	Het LSB nibble wordt het eerst gelezen, daarma het MSB nibble
            	Beide Nibbles worden van plaats verschoven en in de variable "input" gezet
				Deze functie is alleen actief wanner het programma in de Program mode staat
ingang_var		geen
retour_var		globale variabele "input"
*****************************************************
**/
void ReadPositionNibble()
{
	static unsigned char tmp;
	
	// Lees LSB nibble
	ReadNibble();			// Het LSB nibble wordt het eerst verstuurd door Rocrail
	tmp = input;			// en even bewaard in tmp
	// Lees MSB nibble
	ReadNibble();			// Het MSB nibble wordt nu gelezen
	input <<= 3;			// 3 plaatsen naar links geschoven
	input |= tmp;			// en samen gevoegd met het eerdere LSB Nibble
	input <<= 1;
	if (input > 100) { input = 100; }
	// Deze manier geeft uiteindelijk de volgende reeks aan getallen:  0-10-18-28-36-46-54-64-72-82-90-100
}

/**
******************************************************
Functie         void servo_instellen (void)
Beschrijving    Leest de waarden in die van Rocrail afkomen voor het instellen van de servo's en relais
			    De waarden worden door deze functie ook verwerkt en in EEPROM gezet
				Deze functie is alleen actief wanneer het programma in de Program mode staat
ingang_var		geen
retour_var		geen				
*****************************************************
**/

void servo_instellen()
{
	unsigned char tmp_ri;
	static unsigned int tmp;
	
   
	while (programMode)						// Zolang als het programma in de programmeer mode staat
	{
		
		ReadNibble();						// leest 3 bits, de vierde bit is de schrijflijn, variabele "input" heeft nu de nieuwe waarde
		
		if (input == 0)
		{
			// NOP
		}
		else if (input == 0x01)				// Bepalen welk servo nummer wordt gebruikt
		{
			ReadNibble();
			target = input;
		}
		else if (input == 0x02)				// LINKSOM RICHTING INSTELLEN
		{
			ReadPositionNibble();			// Lees de volgende 2 nibble's van 3 bits voor de nieuwe positie voor linksom
			servo[target].linksom = Puls_1_5_msec + (9 * input);
            eeprom_write_word((eeServoLinks_w + (2*target)),servo[target].linksom );		// eindstand links
			if (servo[target].stand <= servo[target].linksom) tmp_ri = 2;		// stand < linksom
			if (servo[target].stand >  servo[target].linksom) tmp_ri = 4;		// stand > linksom
			servo[target].eindstand = 0;
			
			while (servo[target].eindstand == 0)
			{
                Servo_tijden (target,tmp_ri);										// Besturen van de betreffende servo	
				__delay_ms (15);
			}
		}
		
		
		else if (input == 0x03)			// RECHTSOM RICHTING INSTELLEN
		{
			ReadPositionNibble();
			servo[target].rechtsom = Puls_1_5_msec - (9 * input);
            eeprom_write_word((eeServoRechts_w +(2*target)), servo[target].rechtsom);	// eindstand rechts
			if (servo[target].stand <= servo[target].rechtsom) tmp_ri = 1;			// stand < rechtsom
			if (servo[target].stand >  servo[target].rechtsom) tmp_ri = 3;			// stand > rechtsom
			servo[target].eindstand = 0;
			
			while (servo[target].eindstand == 0)
			{	
                Servo_tijden (target,tmp_ri);										// Besturen van de betreffende servo	
                __delay_ms (15);
			}
			
			
		}
			
		else if (input == 0x04)					// RELAIS OMPOLEN
		{
			ReadNibble();
			servo[target].relais = input;                               // input is 0 of 1 
			writeEEprom((eeServoRelais_b + target), input);		// Waarde bewaren in eeprom
            
		}
		else if (input == 0x05)					// SNELHEID SERVO AANPASSEN
		{
			ReadNibble(); // LSB
            tmp = input;
            ReadNibble(); // MSB
            input <<= 3;
            input |= tmp;
            if (input == 0) { input = 1; }
            if (input > 0x0F) { input = 0x0F; }
							
			if (input == 1)		{ servo[target].stapgrootte = 2;	  writeEEprom((eeServoStap_b + target), 2);  }
			if (input == 5)		{ servo[target].stapgrootte = 5;	  writeEEprom((eeServoStap_b + target), 5);  }
			if (input == 13)	{ servo[target].stapgrootte = 8;	  writeEEprom((eeServoStap_b + target), 8);  }
			if (input == 15)	{ servo[target].stapgrootte = 11;	  writeEEprom((eeServoStap_b + target), 11); }
		}
		
		else if (input == 0x06)
		{
			// NOP
		}	
		else
		{
			// Exit programming mode
			programMode = 0;  
			LE1 = 0;                //LE1 OFF
		}
	}
}

/**
*****************************************************
Functie	Servo_uitgang
beschrijving :  Zet de gewenste uitgangen van de servo's hoog
ingang_var		unsigned char nr --> bit-informatie welke uitgang hoog gezet moet worden
retour_var		geen
*****************************************************
**/
void Servo_uitgang_hoog (unsigned char nr)
{
	if (nr == 0) Servo1 = 1;
	if (nr == 1) Servo2 = 1;
	if (nr == 2) Servo3 = 1;
	if (nr == 3) Servo4 = 1;
	
	//	if (nr == 4) dan gebeurt er niets.
}

/**
******************************************************
Functie	Servo_uitgang_laag
beschrijving :  Zet de gewenste uitgangen van de servo's laag
ingang_var		unsigned char nr --> bit-informatie welke uitgang laag gezet moet worden
retour_var		geen

******************************************************
**/
void Servo_uitgang_laag (unsigned char nr)
{
	if (nr == 0) Servo1 = 0;
	if (nr == 1) Servo2 = 0;
	if (nr == 2) Servo3 = 0;
	if (nr == 3) Servo4 = 0;
	
	//	if (nr == (4)) dan gebeurt er niets mee want deze functie doet niets wanneer nr == 4
	//  wanneer nr ==4 aan de beurt is dan is dit de timing van de lage puls, bedoeld om de tijd van 20msec vol te maken.
}

/**
******************************************************
Functie	Schakel_relais_uitgang (unsigned char sv_nr, char relais_stand)
beschrijving :  Schakelt het gewenste relais en houdt daarbij rekening met eventuele ompoling door de gebruiker
				Deze functie bepaalt iedere keer opnieuw de inhoud van 'uitgang' en stuurt die via RS232 naar het GCA137 relaisboard
				De inhoud van servo[] bepaalt dus of een relais wel of niet aan gaat
				Zodra een servo beweegt is zijn eindstand <> 1 en zullen de 2 betreffende bits in 'uitgang' voor die servo op 00 staan
ingang_var		sv_nr, relais_stand
retour_var		geen

******************************************************
**/
void Schakel_relais_uitgang ()
{

	const unsigned char stand_links  [] = {1, 4, 16, 64 };		//	01 01 01 01			binair uitgeschreven, if bit==0 than the relais is off
	const unsigned char stand_rechts [] = {2, 8, 32, 128};		//	10 10 10 10			binair uitgeschreven
	unsigned char temp, test, uitgang, i;
		
    uitgang = 0;                                                        // if all the servos are moving than no relais is switched on
	for (i = 0; i < 4; i++)						  					    // relais standen bepalen voor alle 4 servo's
	{
		if (servo[i].eindstand == 1)									// de servo moet wel op zijn eindpunt staan, anders wordt er geen relais geschakeld
		{ 
			if (servo[i].richting == 0)									// stand van de servo is linksom
			{
				temp = stand_links [i];									// standaard relais voor linksom
				if (servo[i].relais == 1) temp = stand_rechts [i];		// Relais is omgepoold door de gebruiker
				uitgang += temp;
			}

			if (servo[i].richting > 0)									// stand van de servo is rechtsom
			{
				temp = stand_rechts [i];								// standaard relais voor rechtsom
				if (servo[i].relais == 1) temp = stand_links [i];		// Relais is omgepoold door de gebruiker
				uitgang += temp;
			}
		}
	}
    
    while (!TRMT);                                                        // while TransMit buffer is not full
    TXREG = uitgang;                                                      // send data to relais board MGV137
    while (!TRMT);                                                        // do this twice, otherwise the MGV137 will not except the data
    TXREG = uitgang;
    
		// We moeten dezelfde data dus 2x sturen want anders pakt de GCA137 de nieuwe stand van de relais niet over
}

/**
******************************************************
Functie	WaitNS
beschrijving :  Zet het totale programma stil voor het aantal 'nano_sec' dat opgegeven is
				Maakt gebruik van Timer1 zonder interrupt
ingang_var		nano_sec
retour_var		geen

******************************************************
**/
void WaitNS(unsigned int nano_sec)
{
    TMR1IF  = 0;                    // reset overflow bit
    TMR1 = 65535-nano_sec;          // preset TMR1H + TMR1L 
    TMR1ON  = 1;                    // enable timer
    while ( ! TMR1IF)               // Wait till overflow of the timer
        {      
        }
    TMR1ON  = 0;                    // disable timer  
}
	
/*****************************************************
**/
void Feedback (unsigned char sv_nr, unsigned char status)
{
    if (status == 0)
    {
        if (sv_nr == 0)  FEEDBACK1 = 0;
        if (sv_nr == 1)  FEEDBACK2 = 0;
        if (sv_nr == 2)  FEEDBACK3 = 0;
        if (sv_nr == 3)  FEEDBACK4 = 0;
    }    
    if (status == 1)
    {
        if (sv_nr == 0)  FEEDBACK1 = 1;
        if (sv_nr == 1)  FEEDBACK2 = 1;
        if (sv_nr == 2)  FEEDBACK3 = 1;
        if (sv_nr == 3)  FEEDBACK4 = 1;
    }  
}    
/********************************************************
Functie         BounceCycle (sv_nr);
beschrijving :  Puts the Zet de servo's en relais in de goede stand op aangeven van Rocrail
ingang_var		sv_nr, li_re
retour_var		geen

*******************************************************/
void BounceCycle (unsigned char sv_nr, unsigned char li_re)
{
  servo[sv_nr].eindstand = 0;
    
  if ((li_re == 1) || (li_re == 3))       // turn right
  {
      if (servo[sv_nr].bounceCycle == 1)
      {
          servo[sv_nr].stand = servo[sv_nr].rechtsom + 50;
          servo[sv_nr].bounceCycle -= 1;                            // Now bounceCycle == 0 and bouncing will stop.  
      } 
      
      if (servo[sv_nr].bounceCycle == 2)
      {
          servo[sv_nr].stand = servo[sv_nr].rechtsom + 100;
          servo[sv_nr].bounceCycle -= 1;
      }
      
      if (servo[sv_nr].bounceCycle == 3)
      {
          servo[sv_nr].stand = servo[sv_nr].rechtsom + 150;
          servo[sv_nr].bounceCycle -= 1;
      }    
      
  }
}

/**
******************************************************
Functie	Servo_tijden
beschrijving :  Zet de servo's en relais in de goede stand op aangeven van Rocrail
ingang_var		sv_nr, li_re
retour_var		geen

******************************************************
**/
void Servo_tijden (unsigned char sv_nr, unsigned char li_re)
{   
	if (li_re == 1)		// stand < rechtsom
	{
		if ((servo[sv_nr].stand +  servo[sv_nr].stapgrootte) > servo[sv_nr].rechtsom)  { servo[sv_nr].eindstand = 1; }		// eindpunt is bereikt
		else servo[sv_nr].stand += servo[sv_nr].stapgrootte; 
	}
	
	if (li_re == 3)		// stand > rechtsom
	{
		if ((servo[sv_nr].stand -  servo[sv_nr].stapgrootte) < servo[sv_nr].rechtsom)  { servo[sv_nr].eindstand = 1; }		// eindpunt is bereikt
		else servo[sv_nr].stand -= servo[sv_nr].stapgrootte; 		
	}
	
	if (li_re == 2)		// stand < linksom
	{
		if ((servo[sv_nr].stand +  servo[sv_nr].stapgrootte) > servo[sv_nr].linksom)  { servo[sv_nr].eindstand = 1; }		// eindpunt is bereikt
		else servo[sv_nr].stand += servo[sv_nr].stapgrootte ; 
	}
	
	if (li_re == 4)		// stand > linksom
	{
		if ((servo[sv_nr].stand -  servo[sv_nr].stapgrootte) < servo[sv_nr].linksom)  { servo[sv_nr].eindstand = 1; }		// eindpunt is bereikt
		else servo[sv_nr].stand -= servo[sv_nr].stapgrootte; 
	}			
	
	if (servo[sv_nr].eindstand == 0)			// Betreffende servo is nog niet aan zijn eindpunt
	{
		Servo_uitgang_hoog(sv_nr);				// Servo signaal naar 5 Volt
		WaitNS(servo[sv_nr].stand);				// timing hoog signaal
		Servo_uitgang_laag(sv_nr);				// Servo signaal naar 0 Volt
	}	
	
    if ((servo[sv_nr].eindstand == 1) && (servo[sv_nr].bounceCycle >  0))  BounceCycle (sv_nr, li_re);    // the servo is at its end and will bounce several times
    if ((servo[sv_nr].eindstand == 1) && (servo[sv_nr].bounceCycle == 0))                          // the servo is at its end and bounces anymore
	{
		Servo_uitgang_hoog (sv_nr);																   // Uitgang van de betreffende servo naar 5 volt````````````````````````````````````  
		if ((li_re == 1) || (li_re == 3)) writeEEprom ((eeServoRichting_b + sv_nr), 1);            // Laatste richting bewaren, voorkomt plotselinge bewegingen bij inschakelen
		else writeEEprom ((eeServoRichting_b + sv_nr), 0);                                         // (li_re == 2) || (li_re == 4) 			
		eeprom_write_word ((eeServoStand_w + (2*sv_nr)), servo[sv_nr].stand);					   // Laatste stand ook bewaren anders begint de servo maar ergens bij power-up	
		Schakel_relais_uitgang ();																   // Bepaalt en schakelt zonodig alle relais
	}	        
}

int main(void)
{	
	unsigned char i, ri;
    __delay_ms(1000);
	Init();                         // Initialize the IO ports, Timer1, read eeprom etc.  
  
    while(1)
    { 
        if (programMode == 0)												//servo_aansturing();			// stand servo verzetten
        {
            ReadInput();													// ingangen uitlezen, resultaat komt in "input"
            
            for (i = 0; i != 4; i++)        								                                                                
            {
                if (input &  (1 <<  i))                                     // ingang == 1 ??
                    {
                        if (servo[i].stand <= servo[i].rechtsom) ri = 1;	// stand < rechtsom
                        else ri = 3;										// stand > rechtsom                  
  
                        if (servo[i].richting == 0)							// oude stand == linksom
                        {
                            servo[i].richting  = 1;							// nieuwe richting wordt rechtsom
                            servo[i].eindstand = 0;							// servo staat zeker niet op zijn eindstand
                            servo[i].bounceCycle = servo[i].bounce;         // if servo[sv_nr].bounce > 0 then the servo will bounce.
                            Schakel_relais_uitgang ();						// Schakelt in ieder geval de relais uit waarvan de servo[i].eindstand == 0
                        }
                    }
                else														// ingang == 0
                    {
                        if (servo[i].stand <= servo[i].linksom) ri = 2;		// stand < linksom
                        else ri = 4;										// stand > linksom
                    
                        if (servo[i].richting == 1)							// oude stand == rechtsom
                        {													// Wanneer de ingang 0 is en de richting van de servo ook dan gebeurt er dus niets
                            servo[i].richting  = 0;							// nieuwe richting wordt linksom
                            servo[i].eindstand = 0;							// servo staat zeker niet op zijn eindstand
                            servo[i].bounceCycle = servo[i].bounce;         // if servo[sv_nr].bounce > 0 then the servo will bounce.
                            Schakel_relais_uitgang ();						// Schakelt in ieder geval de relais uit waarvan de servo[i].eindstand == 0
                        }
                }
                if (servo[i].eindstand == 0) 								// Betreffende servo is nog niet aan zijn eindpunt
                {
                    Servo_tijden(i,ri);									    // servo's 1 voor 1 aansturen	
                }
                
                if (servo[i].eindstand == 1)                                // de servo staat dus op zijn eindstand
                {
                    Servo_uitgang_hoog (i);									// Uitgang van de betreffende servo naar 5 volt
                    Feedback (i, servo[i].richting);                        // switch on/off output feedback
                }
                    
            }   // for (i = 0; i != 4; i++) 
              __delay_ms (12);
        }
    //---------------------------------------------------------------------------------------------------------------------						
        if (programMode == 1)
        {
            servo_instellen();			// servo's instellen
        }					
	}  
}

